﻿using System.Collections.Generic;
using UnityEngine;

/// <summary>
/// 高效A*寻路器
/// 特点：
/// 1. 使用二叉堆优先队列，Open列表操作复杂度从O(n)降到O(log n)
/// 2. 使用字典存储节点数据，避免在节点类中添加寻路字段
/// 3. 使用HashSet实现O(1)的Closed列表查询
/// 4. 支持路径平滑
/// </summary>
public class AStarPathfinder
{
    /// <summary>
    /// 寻路节点数据（避免污染OctreeTreeNode类）
    /// </summary>
    private class PathNode
    {
        public OctreeTreeNode Node;
        public PathNode Parent;
        public float G; // 从起点到当前节点的实际代价
        public float H; // 从当前节点到终点的启发式估计
        public float F => G + H; // 总代价

        public PathNode(OctreeTreeNode node)
        {
            Node = node;
            Parent = null;
            G = float.MaxValue;
            H = 0;
        }
    }

    /// <summary>
    /// 二叉堆优先队列（最小堆）
    /// </summary>
    private class BinaryHeap
    {
        private List<PathNode> heap = new List<PathNode>();
        private Dictionary<OctreeTreeNode, int> indexMap = new Dictionary<OctreeTreeNode, int>();

        public int Count => heap.Count;

        public void Clear()
        {
            heap.Clear();
            indexMap.Clear();
        }

        public bool Contains(OctreeTreeNode node)
        {
            return indexMap.ContainsKey(node);
        }

        public void Enqueue(PathNode node)
        {
            heap.Add(node);
            int index = heap.Count - 1;
            indexMap[node.Node] = index;
            BubbleUp(index);
        }

        public PathNode Dequeue()
        {
            if (heap.Count == 0) return null;

            PathNode min = heap[0];
            indexMap.Remove(min.Node);

            int lastIndex = heap.Count - 1;
            if (lastIndex > 0)
            {
                heap[0] = heap[lastIndex];
                indexMap[heap[0].Node] = 0;
                heap.RemoveAt(lastIndex);
                BubbleDown(0);
            }
            else
            {
                heap.RemoveAt(0);
            }

            return min;
        }

        /// <summary>
        /// 当节点的F值更新后，重新调整堆
        /// </summary>
        public void UpdatePriority(PathNode node)
        {
            if (!indexMap.TryGetValue(node.Node, out int index)) return;
            BubbleUp(index);
        }

        private void BubbleUp(int index)
        {
            while (index > 0)
            {
                int parentIndex = (index - 1) / 2;
                if (heap[index].F >= heap[parentIndex].F) break;

                Swap(index, parentIndex);
                index = parentIndex;
            }
        }

        /// <summary>
        /// 重新构建二叉堆，从最后一个非叶子节点开始，向下比较
        /// </summary>
        /// <param name="index"></param>
        private void BubbleDown(int index)
        {
            int count = heap.Count;
            while (true)
            {
                int smallest = index;
                int left = 2 * index + 1;
                int right = 2 * index + 2;

                if (left < count && heap[left].F < heap[smallest].F)
                    smallest = left;
                if (right < count && heap[right].F < heap[smallest].F)
                    smallest = right;

                if (smallest == index) break;

                Swap(index, smallest);
                index = smallest;
            }
        }

        private void Swap(int i, int j)
        {
            PathNode temp = heap[i];
            heap[i] = heap[j];
            heap[j] = temp;

            indexMap[heap[i].Node] = i;
            indexMap[heap[j].Node] = j;
        }
    }

    private MyOctreetree octree;
    private BinaryHeap openList;
    private HashSet<OctreeTreeNode> closedSet;
    private Dictionary<OctreeTreeNode, PathNode> nodeDataMap;

    // 性能统计
    public int NodesExplored { get; private set; }
    public float LastSearchTimeMs { get; private set; }

    public AStarPathfinder(MyOctreetree octree)
    {
        this.octree = octree;
        openList = new BinaryHeap();
        closedSet = new HashSet<OctreeTreeNode>();
        nodeDataMap = new Dictionary<OctreeTreeNode, PathNode>();
    }

    /// <summary>
    /// 寻找从起点到终点的路径
    /// </summary>
    /// <param name="startPos">起点世界坐标</param>
    /// <param name="endPos">终点世界坐标</param>
    /// <param name="smooth">是否平滑路径</param>
    /// <returns>路径点列表（世界坐标），找不到返回null</returns>
    public List<Vector3> FindPath(Vector3 startPos, Vector3 endPos, bool smooth = false)
    {
        var stopwatch = System.Diagnostics.Stopwatch.StartNew();
        NodesExplored = 0;

        // 找到起点和终点所在的八叉树节点
        OctreeTreeNode startNode = FindNodeContainingPoint(startPos);
        OctreeTreeNode endNode = FindNodeContainingPoint(endPos);

        if (startNode == null || endNode == null)
        {
            Debug.LogWarning("起点或终点不在可通行区域内");
            stopwatch.Stop();
            LastSearchTimeMs = stopwatch.ElapsedTicks / (float)System.Diagnostics.Stopwatch.Frequency * 1000f;
            return null;
        }

        if (startNode == endNode)
        {
            stopwatch.Stop();
            LastSearchTimeMs = stopwatch.ElapsedTicks / (float)System.Diagnostics.Stopwatch.Frequency * 1000f;
            return new List<Vector3> { startPos, endPos };
        }

        // 清理上次搜索的数据
        openList.Clear();
        closedSet.Clear();
        nodeDataMap.Clear();

        // 初始化起点
        PathNode startPathNode = GetOrCreatePathNode(startNode);
        startPathNode.G = 0;
        startPathNode.H = Heuristic(startNode, endNode);
        openList.Enqueue(startPathNode);

        while (openList.Count > 0)
        {
            PathNode current = openList.Dequeue();
            NodesExplored++;

            // 到达终点
            if (current.Node == endNode)
            {
                var path = ReconstructPath(current, startPos, endPos);
                stopwatch.Stop();
                LastSearchTimeMs = stopwatch.ElapsedTicks / (float)System.Diagnostics.Stopwatch.Frequency * 1000f;

                if (smooth)
                {
                    path = SmoothPath(path);
                }

                return path;
            }

            closedSet.Add(current.Node);

            // 遍历邻居
            foreach (var neighborNode in current.Node.neighbors)
            {
                if (closedSet.Contains(neighborNode)) continue;

                float tentativeG = current.G + GetMovementCost(current.Node, neighborNode);

                PathNode neighborPathNode = GetOrCreatePathNode(neighborNode);
                bool inOpenList = openList.Contains(neighborNode);

                if (tentativeG < neighborPathNode.G)
                {
                    neighborPathNode.Parent = current;
                    neighborPathNode.G = tentativeG;
                    neighborPathNode.H = Heuristic(neighborNode, endNode);

                    if (inOpenList)
                    {
                        openList.UpdatePriority(neighborPathNode);
                    }
                    else
                    {
                        openList.Enqueue(neighborPathNode);
                    }
                }
            }
        }

        // 找不到路径
        stopwatch.Stop();
        LastSearchTimeMs = stopwatch.ElapsedTicks / (float)System.Diagnostics.Stopwatch.Frequency * 1000f;
        Debug.LogWarning($"找不到从 {startPos} 到 {endPos} 的路径");
        return null;
    }

    /// <summary>
    /// 获取或创建PathNode
    /// </summary>
    private PathNode GetOrCreatePathNode(OctreeTreeNode node)
    {
        if (!nodeDataMap.TryGetValue(node, out PathNode pathNode))
        {
            pathNode = new PathNode(node);
            nodeDataMap[node] = pathNode;
        }
        return pathNode;
    }

    /// <summary>
    /// 启发式函数 - 欧几里得距离
    /// </summary>
    private float Heuristic(OctreeTreeNode from, OctreeTreeNode to)
    {
        return Vector3.Distance(from.nodeCube.center, to.nodeCube.center);
    }

    /// <summary>
    /// 获取两个相邻节点之间的移动代价
    /// </summary>
    private float GetMovementCost(OctreeTreeNode from, OctreeTreeNode to)
    {
        // 基础代价：欧几里得距离
        float distance = Vector3.Distance(from.nodeCube.center, to.nodeCube.center);
        return distance;
    }

    /// <summary>
    /// 在八叉树中找到包含指定点的可通行叶子节点
    /// </summary>
    private OctreeTreeNode FindNodeContainingPoint(Vector3 point)
    {
        if (octree?.leafNodes == null) return null;

        OctreeTreeNode bestNode = null;
        float bestDistance = float.MaxValue;

        foreach (var node in octree.leafNodes)
        {
            if (node.IsWalkable && node.nodeCube.Contains(point))
            {
                return node;
            }

            // 如果点不在任何节点内，找最近的可通行节点
            if (node.IsWalkable)
            {
                float dist = Vector3.Distance(point, node.nodeCube.center);
                if (dist < bestDistance)
                {
                    bestDistance = dist;
                    bestNode = node;
                }
            }
        }

        return bestNode;
    }

    /// <summary>
    /// 从终点回溯重建路径
    /// </summary>
    private List<Vector3> ReconstructPath(PathNode endPathNode, Vector3 startPos, Vector3 endPos)
    {
        List<Vector3> path = new List<Vector3>();

        PathNode current = endPathNode;
        while (current != null)
        {
            path.Add(current.Node.nodeCube.center);
            current = current.Parent;
        }

        path.Reverse();

        // 用实际的起点和终点替换首尾节点中心
        if (path.Count > 0)
        {
            path[0] = startPos;
            path[path.Count - 1] = endPos;
        }

        return path;
    }

    /// <summary>
    /// 路径平滑 - 使用简单的拉直算法
    /// </summary>
    private List<Vector3> SmoothPath(List<Vector3> path)
    {
        if (path == null || path.Count <= 2) return path;

        List<Vector3> smoothed = new List<Vector3> { path[0] };
        int current = 0;

        while (current < path.Count - 1)
        {
            int furthest = current + 1;

            for (int i = path.Count - 1; i > current + 1; i--)
            {
                if (CanWalkDirect(path[current], path[i]))
                {
                    furthest = i;
                    break;
                }
            }

            smoothed.Add(path[furthest]);
            current = furthest;
        }

        return smoothed;
    }

    /// <summary>
    /// 检查两点之间是否可以直接通行
    /// </summary>
    private bool CanWalkDirect(Vector3 from, Vector3 to)
    {
        Vector3 direction = to - from;
        float distance = direction.magnitude;

        if (Physics.Raycast(from, direction.normalized, distance))
        {
            return false;
        }

        return true;
    }
}

/// <summary>
/// A*寻路测试组件
/// </summary>
public class AStarPathfinderTest : MonoBehaviour
{
    [Header("寻路设置")]
    public Transform startPoint;
    public Transform endPoint;
    public bool smoothPath = true;

    [Header("调试绘制")]
    public bool drawPath = true;
    public Color pathColor = Color.cyan;
    public float pathWidth = 0.1f;

    [Header("统计信息（只读）")]
    [SerializeField] private int nodesExplored;
    [SerializeField] private float searchTimeMs;
    [SerializeField] private int pathLength;

    private AStarPathfinder pathfinder;
    private List<Vector3> currentPath;
    private CreatOctreeTree octreeCreator;
    private bool isInitialized = false;

    /// <summary>
    /// 获取探索节点数
    /// </summary>
    public int NodesExplored => nodesExplored;

    /// <summary>
    /// 获取搜索时间(ms)
    /// </summary>
    public float SearchTimeMs => searchTimeMs;

    /// <summary>
    /// 获取路径长度
    /// </summary>
    public int PathLength => pathLength;

    /// <summary>
    /// 是否已初始化
    /// </summary>
    public bool IsInitialized => isInitialized;

    private void Start()
    {
        InitializePathfinder();
    }

    /// <summary>
    /// 初始化寻路器
    /// </summary>
    public void InitializePathfinder()
    {
        octreeCreator = FindObjectOfType<CreatOctreeTree>();
        if (octreeCreator == null)
        {
            Debug.LogWarning("未找到 CreatOctreeTree 组件");
            return;
        }

        var octreeField = typeof(CreatOctreeTree).GetField("myOctreetree",
            System.Reflection.BindingFlags.NonPublic | System.Reflection.BindingFlags.Instance);

        if (octreeField != null)
        {
            var octree = octreeField.GetValue(octreeCreator) as MyOctreetree;
            if (octree != null)
            {
                pathfinder = new AStarPathfinder(octree);
                isInitialized = true;
                Debug.Log("A*寻路器初始化完成");
            }
        }
    }

    /// <summary>
    /// 执行寻路（供Editor调用）
    /// </summary>
    public void ExecuteFindPath()
    {
        if (!isInitialized)
        {
            InitializePathfinder();
        }

        if (pathfinder == null)
        {
            Debug.LogWarning("寻路器未初始化");
            return;
        }

        if (startPoint == null || endPoint == null)
        {
            Debug.LogWarning("请设置起点和终点 Transform");
            return;
        }

        currentPath = pathfinder.FindPath(startPoint.position, endPoint.position, smoothPath);
        UpdateStats();
    }

    /// <summary>
    /// 清除当前路径
    /// </summary>
    public void ClearPath()
    {
        currentPath = null;
        nodesExplored = 0;
        searchTimeMs = 0;
        pathLength = 0;
        Debug.Log("路径已清除");
    }

    private void UpdateStats()
    {
        if (pathfinder != null)
        {
            nodesExplored = pathfinder.NodesExplored;
            searchTimeMs = pathfinder.LastSearchTimeMs;
        }
        pathLength = currentPath?.Count ?? 0;

        if (currentPath != null)
        {
            Debug.Log($"寻路完成！探索节点: {nodesExplored}, 耗时: {searchTimeMs:F3}ms, 路径点数: {pathLength}");
        }
    }

    private void OnDrawGizmos()
    {
        if (!drawPath || currentPath == null || currentPath.Count < 2) return;

        Gizmos.color = pathColor;
        for (int i = 0; i < currentPath.Count - 1; i++)
        {
            Gizmos.DrawLine(currentPath[i], currentPath[i + 1]);
            Gizmos.DrawWireSphere(currentPath[i], pathWidth);
        }
        Gizmos.DrawWireSphere(currentPath[currentPath.Count - 1], pathWidth);
    }
}

